'use client';

import { useState, useEffect, Suspense } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import { useDatabaseContext } from '@/lib/context/DatabaseContext';
import DatabaseSelector from '@/components/DatabaseSelector';
import NavBar from '@/components/NavBar';
import { FieldConfig } from '@/lib/config/databases';

interface FieldEditabilityConfig {
  fieldName: string;
  displayName: string;
  isEditable: boolean;
  isHidden: boolean;
}

// Main component that will be wrapped in Suspense
function AdminContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const { 
    currentHostId, 
    currentDatabaseName, 
    currentTableName,
    currentTable,
    availableTables,
    setCurrentHostId
  } = useDatabaseContext();
  
  // Get host from URL params if present
  useEffect(() => {
    const hostParam = searchParams.get('host');
    console.log('[Admin] URL host param:', hostParam, 'Current host ID:', currentHostId);
    
    if (hostParam && hostParam !== currentHostId) {
      console.log(`[Admin] Switching to host from URL: ${hostParam}`);
      setCurrentHostId(hostParam);
    } else if (!hostParam && currentHostId !== 'host1') {
      console.log('[Admin] No host in URL, defaulting to host1');
      setCurrentHostId('host1');
    }
  }, [searchParams, currentHostId, setCurrentHostId]);
  
  // Log context changes
  useEffect(() => {
    console.log('[Admin] Context updated:', {
      currentHostId,
      currentDatabaseName,
      currentTableName,
      searchParams: searchParams.toString()
    });
  }, [currentHostId, currentDatabaseName, currentTableName, searchParams]);
  
  // Log current table configuration when it changes
  useEffect(() => {
    if (currentTable) {
      console.log(`[Admin] Current table config for ${currentHostId}/${currentDatabaseName}/${currentTableName}:`, {
        tableName: currentTable.tableName,
        editableFields: currentTable.editableFields,
        readOnlyFields: currentTable.readOnlyFields,
        fields: currentTable.fields.map(f => ({
          fieldName: f.fieldName,
          displayName: f.displayName,
          type: f.type,
          readOnly: f.readOnly,
          hidden: f.hidden
        }))
      });
    }
  }, [currentTable, currentHostId, currentDatabaseName, currentTableName]);
  
  const [fields, setFields] = useState<FieldEditabilityConfig[]>([]);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [saveSuccess, setSaveSuccess] = useState(false);
  
  // Load fields from the current table configuration and fetch actual fields from the database
  useEffect(() => {
    if (!currentTable || !currentHostId || !currentDatabaseName || !currentTableName) {
      setFields([]);
      setLoading(false);
      return;
    }
    
    setLoading(true);
    
    const fetchTableMetadata = async () => {
      try {
        // Get existing field configurations
        const editableFields = currentTable.editableFields || [];
        const hiddenFields = currentTable.hiddenFields || [];
        const readOnlyFields = currentTable.readOnlyFields || [];
        
        console.log(`[Admin] Fetching metadata for ${currentHostId}/${currentDatabaseName}/${currentTableName}`);
        console.log(`[Admin] Configured read-only fields:`, readOnlyFields);
        console.log(`[Admin] Configured editable fields:`, editableFields);
        
        // First, try to get a sample record to see all available fields
        const response = await fetch(`/api/odata/${currentHostId}/${currentDatabaseName}/${currentTableName}?$top=1`);
        
        if (!response.ok) {
          console.error(`[Admin] Failed to fetch table metadata:`, response.status, response.statusText);
          throw new Error(`Failed to fetch table metadata: ${response.statusText}`);
        }
        
        const data = await response.json();
        const records = data.value || [];
        
        // Get all field names from the first record
        let allFieldNames: string[] = [];
        
        if (records.length > 0) {
          // Extract field names from the record, excluding OData metadata fields
          allFieldNames = Object.keys(records[0]).filter(key => !key.startsWith('@odata'));
        }
        
        // Combine fields from the configuration and the actual record
        const configFieldNames = currentTable.fields.map(field => field.fieldName);
        const combinedFieldNames = [...new Set([...configFieldNames, ...allFieldNames])];
        
        // Create field configurations
        const fieldConfigs = combinedFieldNames.map(fieldName => {
          // Find field in configuration if it exists
          const configField = currentTable.fields.find(f => f.fieldName === fieldName);
          
          // Determine if field is read-only from configuration
          const isReadOnly = configField?.readOnly || readOnlyFields.includes(fieldName);
          const isEditable = !isReadOnly && (editableFields.includes(fieldName) || !readOnlyFields.includes(fieldName));
          
          return {
            fieldName,
            displayName: configField?.displayName || formatFieldName(fieldName),
            isEditable,
            isHidden: hiddenFields.includes(fieldName) || configField?.hidden || false
          };
        });
        
        setFields(fieldConfigs);
        setError(null);
      } catch (err) {
        console.error('Error loading field configuration:', err);
        setError(err instanceof Error ? err.message : 'An unknown error occurred');
      } finally {
        setLoading(false);
      }
    };
    
    fetchTableMetadata();
  }, [currentTable, currentHostId, currentDatabaseName, currentTableName]);
  
  // Format field name for display (e.g., 'firstName' -> 'First Name')
  const formatFieldName = (field: string): string => {
    return field
      .replace(/([A-Z])/g, ' $1')
      .replace(/^./, str => str.toUpperCase())
      .trim();
  };
  
  // Handle checkbox change for field editability
  const handleEditableChange = (fieldName: string, isEditable: boolean) => {
    setFields(prevFields => 
      prevFields.map(field => {
        // Don't allow changing editability of read-only fields from config
        const configField = currentTable?.fields.find(f => f.fieldName === fieldName);
        const isReadOnly = configField?.readOnly || currentTable?.readOnlyFields?.includes(fieldName);
        
        if (isReadOnly) {
          console.log(`[Admin] Cannot modify read-only field: ${fieldName}`);
          return field;
        }
        
        return field.fieldName === fieldName 
          ? { ...field, isEditable } 
          : field;
      })
    );
  };
  
  // Handle checkbox change for field hiding
  const handleHiddenChange = (fieldName: string, isHidden: boolean) => {
    setFields(prevFields => 
      prevFields.map(field => 
        field.fieldName === fieldName 
          ? { ...field, isHidden } 
          : field
      )
    );
  };
  
  // Handle form submission to save configuration
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setSaving(true);
    setError(null);
    setSaveSuccess(false);
    
    try {
      // Extract editable field names
      const editableFieldNames = fields
        .filter(field => field.isEditable && !field.isHidden)
        .map(field => field.fieldName);
      
      // Extract read-only field names
      const readOnlyFieldNames = fields
        .filter(field => !field.isEditable && !field.isHidden)
        .map(field => field.fieldName);
      
      // Get hidden fields
      const hiddenFields = fields
        .filter(field => field.isHidden)
        .map(field => field.fieldName);
      
      // Create the configuration object to save
      const configToSave = {
        hostId: currentHostId,
        databaseName: currentDatabaseName,
        tableName: currentTableName,
        editableFields: editableFieldNames,
        readOnlyFields: readOnlyFieldNames,
        hiddenFields,
      };
      
      // Save the configuration
      const response = await fetch('/api/admin/field-config', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(configToSave),
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || `Error ${response.status}: ${response.statusText}`);
      }
      
      setSaveSuccess(true);
      
      // Reload the page after a short delay to reflect the changes
      setTimeout(() => {
        window.location.reload();
      }, 1500);
      
    } catch (err) {
      console.error('Error saving field configuration:', err);
      setError(err instanceof Error ? err.message : 'An unknown error occurred');
    } finally {
      setSaving(false);
    }
  };
  
  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50">
        <NavBar />
        
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="bg-white shadow rounded-lg p-6">
              <h1 className="text-2xl font-bold mb-6">Admin Dashboard</h1>
              <div className="animate-pulse flex space-x-4">
                <div className="flex-1 space-y-6 py-1">
                  <div className="h-4 bg-gray-200 rounded w-3/4"></div>
                  <div className="space-y-3">
                    <div className="grid grid-cols-3 gap-4">
                      <div className="h-4 bg-gray-200 rounded col-span-2"></div>
                      <div className="h-4 bg-gray-200 rounded col-span-1"></div>
                    </div>
                    <div className="h-4 bg-gray-200 rounded"></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }
  
  return (
    <div className="min-h-screen bg-gray-100">
      <NavBar />
      <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
        <div className="bg-white rounded-lg shadow-lg overflow-hidden">
          <div className="px-3 sm:px-6 py-4 border-b border-gray-200">
            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-3">
              <h1 className="text-xl font-semibold text-gray-900">
                Field Editability Configuration
              </h1>
            </div>
          </div>
          
          <div className="p-1 sm:p-3 md:p-4">
            <p className="text-gray-600 mb-4">
              Select a host, database, and table, then configure which fields should be editable.
            </p>
            
            <DatabaseSelector />
            
            {error && (
              <div className="bg-red-50 border-l-4 border-red-400 p-4 my-4">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-red-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.28 7.22a.75.75 0 00-1.06 1.06L8.94 10l-1.72 1.72a.75.75 0 101.06 1.06L10 11.06l1.72 1.72a.75.75 0 101.06-1.06L11.06 10l1.72-1.72a.75.75 0 00-1.06-1.06L10 8.94 8.28 7.22z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-red-700">{error}</p>
                  </div>
                </div>
              </div>
            )}
            
            {saveSuccess && (
              <div className="bg-green-50 border-l-4 border-green-400 p-4 my-4">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-green-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-green-700">Configuration saved successfully! Reloading...</p>
                  </div>
                </div>
              </div>
            )}
            
            {currentTable ? (
              <form onSubmit={handleSubmit} className="mt-6">
                <div className="overflow-x-auto">
                  <table className="min-w-full divide-y divide-gray-300">
                    <thead className="bg-gray-50">
                      <tr>
                        <th scope="col" className="py-3.5 pl-4 pr-3 text-left text-sm font-semibold text-gray-900 sm:pl-6">Field Name</th>
                        <th scope="col" className="px-3 py-3.5 text-left text-sm font-semibold text-gray-900">Display Name</th>
                        <th scope="col" className="px-3 py-3.5 text-center text-sm font-semibold text-gray-900">Editable</th>
                        <th scope="col" className="px-3 py-3.5 text-center text-sm font-semibold text-gray-900">Omit</th>
                      </tr>
                    </thead>
                    <tbody className="divide-y divide-gray-200 bg-white">
                      {fields.map((field, index) => (
                        <tr key={field.fieldName}>
                          <td className="whitespace-nowrap py-4 pl-4 pr-3 text-sm font-medium text-gray-900 sm:pl-6">
                            {field.fieldName}
                          </td>
                          <td className="whitespace-nowrap px-3 py-4 text-sm text-gray-500">
                            {field.displayName}
                          </td>
                          <td className="whitespace-nowrap px-3 py-4 text-sm text-gray-500 text-center">
                            <input
                              type="checkbox"
                              checked={field.isEditable}
                              onChange={(e) => handleEditableChange(field.fieldName, e.target.checked)}
                              className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                            />
                          </td>
                          <td className="whitespace-nowrap px-3 py-4 text-sm text-gray-500 text-center">
                            <input
                              type="checkbox"
                              checked={field.isHidden}
                              onChange={(e) => handleHiddenChange(field.fieldName, e.target.checked)}
                              className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                            />
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
                
                <div className="flex justify-end pt-6">
                  <button
                    type="submit"
                    className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-opacity-50"
                    disabled={saving}
                  >
                    {saving ? 'Saving...' : 'Save Configuration'}
                  </button>
                </div>
              </form>
            ) : (
              <div className="text-center py-10">
                <p className="text-gray-500">Please select a host, database, and table to configure fields.</p>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}

// Export the main component wrapped in Suspense
export default function AdminPage() {
  return (
    <Suspense fallback={
      <div className="min-h-screen bg-gray-50">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="bg-white shadow rounded-lg p-6">
              <h1 className="text-2xl font-bold mb-6">Admin Dashboard</h1>
              <p>Loading admin dashboard...</p>
            </div>
          </div>
        </div>
      </div>
    }>
      <AdminContent />
    </Suspense>
  );
}
