'use client';

import { useState, useEffect } from 'react';
import { useRouter } from 'next/navigation';
import Link from 'next/link';
import { createODataClient } from '@/lib/api/odataClient';

const LoginPage = () => {
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [error, setError] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [rememberMe, setRememberMe] = useState(false);
  const [success, setSuccess] = useState<{name: string; id: string} | null>(null);
  const router = useRouter();

  // Load saved login ID from localStorage when component mounts
  useEffect(() => {
    if (typeof window !== 'undefined') {
      const savedLoginId = localStorage.getItem('login-id');
      if (savedLoginId) {
        setEmail(savedLoginId);
        setRememberMe(true);
      }
    }
  }, []);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);
    setError('');
    setSuccess(null);

    try {
      if (!email) {
        setError('Please enter your login ID');
        return;
      }

      // Save login ID to appropriate storage based on remember me setting
      if (typeof window !== 'undefined') {
        if (rememberMe) {
          localStorage.setItem('login-id', email);
        } else {
          localStorage.removeItem('login-id');
        }
        sessionStorage.setItem('login-id', email);
      }
      
      // Initialize OData client (using host1 as default)
      const odataClient = createODataClient('host1');
      
      try {
        // Query the Clients table to verify the login ID and password
        const url = `/api/odata/revplan/Clients?$filter=Name_id eq ${email} and web_pass eq '${password}'`;
        console.log('Login API request:', url);
        
        const response = await fetch(url);
        const result = await response.json();
        
        console.log('Login API response:', result);
        
        // The response has a data array containing the results
        if (result.data && result.data.length > 0) {
          // Client found with matching credentials
          const clientData = result.data[0];
          const clientId = clientData.Name_id;
          const clientName = clientData.Name || 'Client';
          
          console.log('Client found, will redirect to client page with ID:', clientId);
          setSuccess({
            name: clientName,
            id: clientId
          });
          
          // Save client name to session storage
          if (typeof window !== 'undefined') {
            sessionStorage.setItem('login-name', clientName);
          }
          
          // Redirect after 2 seconds
          setTimeout(() => {
            router.push('/clients-all');
          }, 2000);
        } else {
          console.log('Invalid credentials for login ID:', email);
          setError('Invalid login ID or password');
        }
      } catch (err) {
        console.error('Error searching for client:', err);
        setError('Error verifying login. Please try again.');
      }
    } catch (err) {
      setError('Invalid credentials. Please try again.');
      console.error('Login error:', err);
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="min-h-screen flex items-center justify-center bg-gray-50 py-12 px-4 sm:px-6 lg:px-8">
      <div className="max-w-md w-full space-y-8">
        <div>
          <h2 className="mt-6 text-center text-3xl font-extrabold text-gray-900">
            Sign in to your account
          </h2>
          <p className="mt-2 text-center text-sm text-gray-600">
            Or{' '}
            <Link href="/" className="font-medium text-indigo-600 hover:text-indigo-500">
              continue as guest
            </Link>
          </p>
        </div>
        
        {success && (
          <div className="mb-8">
            <div className="bg-green-50 border border-green-200 text-green-800 px-6 py-4 rounded-lg shadow-sm" role="alert">
              <p className="text-2xl font-bold mb-2">Welcome back, {success.name}!</p>
              <p className="text-lg text-green-700 mb-3">ID: {success.id}</p>
              <p className="text-base text-green-600">Redirecting to your dashboard...</p>
            </div>
          </div>
        )}
        {error && (
          <div className="mb-6">
            <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded relative" role="alert">
              <span className="block sm:inline">{error}</span>
            </div>
          </div>
        )}
        
        <form className="mt-8 space-y-6" onSubmit={handleSubmit}>
          <input type="hidden" name="remember" value="true" />
          <div className="rounded-md shadow-sm -space-y-px">
            <div>
              <label htmlFor="email-address" className="sr-only">
                Login ID
              </label>
              <input
                id="login-id"
                name="loginId"
                type="text"
                autoComplete="username"
                required
                className="appearance-none rounded-none relative block w-full px-3 py-2 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-t-md focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 focus:z-10 sm:text-sm"
                placeholder="Login ID"
                value={email}
                onChange={(e) => setEmail(e.target.value.trim())}
                disabled={isLoading}
                minLength={1}
              />
            </div>
            <div>
              <label htmlFor="password" className="sr-only">
                Password
              </label>
              <input
                id="password"
                name="password"
                type="password"
                autoComplete="current-password"
                required
                className="appearance-none rounded-none relative block w-full px-3 py-2 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-b-md focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 focus:z-10 sm:text-sm"
                placeholder="Password"
                value={password}
                onChange={(e) => setPassword(e.target.value)}
                disabled={isLoading}
              />
            </div>
          </div>

          <div className="flex items-center justify-between">
            <div className="flex items-center">
              <input
                id="remember-me"
                name="remember-me"
                type="checkbox"
                checked={rememberMe}
                onChange={(e) => setRememberMe(e.target.checked)}
                className="h-4 w-4 text-indigo-600 focus:ring-indigo-500 border-gray-300 rounded"
              />
              <label htmlFor="remember-me" className="ml-2 block text-sm text-gray-900">
                Remember me
              </label>
            </div>

            <div className="text-sm">
              <a href="#" className="font-medium text-indigo-600 hover:text-indigo-500">
                Forgot your password?
              </a>
            </div>
          </div>

          <div>
            <button
              type="submit"
              disabled={isLoading}
              className="group relative w-full flex justify-center py-2 px-4 border border-transparent text-sm font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isLoading ? (
                <>
                  <svg className="animate-spin -ml-1 mr-3 h-5 w-5 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                    <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                    <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                  </svg>
                  Signing in...
                </>
              ) : 'Sign in'}
            </button>
          </div>
        </form>
        
        <div className="mt-6">
          <div className="relative">
            <div className="absolute inset-0 flex items-center">
              <div className="w-full border-t border-gray-300"></div>
            </div>
            <div className="relative flex justify-center text-sm">
              <span className="px-2 bg-gray-50 text-gray-500">
                Don't have an account?
              </span>
            </div>
          </div>

          <div className="mt-6">
            <Link
              href="/register"
              className="w-full flex items-center justify-center px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50"
            >
              Create new account
            </Link>
          </div>
        </div>
      </div>
    </div>
  );
};

export default LoginPage;
