'use client';

import { useState, useEffect, Suspense } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import Link from 'next/link';
import { useDatabaseContext } from '@/lib/context/DatabaseContext';
import NavBar from '@/components/NavBar';
import { statusOptions, priorityOptions, categoryOptions, noteTypeOptions, topicOptions } from '@/lib/config/field-options';

// Import mobile styles
import './mobile-styles.css';

interface RecordData {
  [key: string]: any;
}

interface FormField {
  key: string;
  value: any;
  type: string;
  required: boolean;
  displayName?: string;
  options?: string[];
  multiline?: boolean;
  readOnly?: boolean;
  step?: string;
  inputmode?: string;
  rows?: number;
}

// Main content component that uses useSearchParams
function TaskUpdateContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const { 
    currentHostId, 
    currentDatabaseName, 
    currentTableName,
    currentTable,
    setCurrentHostId,
    setCurrentDatabaseName,
    setCurrentTableName
  } = useDatabaseContext();
  
  const [record, setRecord] = useState<RecordData | null>(null);
  const [formFields, setFormFields] = useState<FormField[]>([]);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [saveSuccess, setSaveSuccess] = useState(false);
  
  // State for client name and ID from session storage
  const [clientName, setClientName] = useState<string>('');
  const [storedClientId, setStoredClientId] = useState<string>('');
  
  // Load client name and ID from session storage on component mount
  useEffect(() => {
    // Need to check if window is defined since this is a Next.js app with SSR
    if (typeof window !== 'undefined') {
      const storedName = sessionStorage.getItem('cl-name') || '';
      const storedId = sessionStorage.getItem('clid') || '';
      
      setClientName(storedName);
      setStoredClientId(storedId);
      
      console.log('Retrieved from session storage:', {
        'cl-name': storedName,
        'clid': storedId
      });
    }
  }, []);
  
  // Get parameters from URL
  const recordId = searchParams ? searchParams.get('tid') : null;
  const hostId = searchParams ? searchParams.get('host') : null;
  const databaseName = searchParams ? searchParams.get('database') : null;
  const tableName = searchParams ? searchParams.get('table') : null;
  
  // Set the context values from URL parameters if provided
  useEffect(() => {
    if (hostId) setCurrentHostId(hostId);
    if (databaseName) setCurrentDatabaseName(databaseName);
    if (tableName) setCurrentTableName(tableName);
  }, [hostId, databaseName, tableName, setCurrentHostId, setCurrentDatabaseName, setCurrentTableName]);

  // Fetch record data when component mounts or when parameters change
  useEffect(() => {
    const fetchRecord = async () => {
      if (!currentHostId || !currentDatabaseName || !currentTableName) {
        setError('Please select a host, database, and table');
        setLoading(false);
        return;
      }
      
      // If no record ID, we're creating a new record
      if (!recordId) {
        setRecord({});
        convertRecordToFormFields({});
        setLoading(false);
        return;
      }

      try {
        setLoading(true);
        setError(null);
        setSaveSuccess(false);
        
        // Build the URL according to our API route's expected format
        const encodedDbName = encodeURIComponent(currentDatabaseName);
        const encodedTableName = encodeURIComponent(currentTableName);
        const encodedRecordId = encodeURIComponent(recordId);
        const response = await fetch(`/api/odata/${encodedDbName}/${encodedTableName}/${encodedRecordId}`, {
          headers: {
            'X-Host-Id': currentHostId,
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'OData-Version': '4.0',
            'OData-MaxVersion': '4.0'
          },
          cache: 'no-store' // Prevent caching of the response
        });
        
        if (!response.ok) {
          throw new Error(`Failed to fetch record: ${response.statusText}`);
        }
        
        const data = await response.json();
        setRecord(data);
        convertRecordToFormFields(data);
      } catch (err) {
        console.error('Error fetching record:', err);
        setError(err instanceof Error ? err.message : 'An unknown error occurred');
      } finally {
        setLoading(false);
      }
    };

    fetchRecord();
  }, [recordId, currentHostId, currentDatabaseName, currentTableName]);

  // Get editable and read-only fields from table configuration or use defaults
  const editableFields = currentTable?.editableFields || ['Note', 'Pri', 'Status', 'DateDue', 'pl_Topic'];
  const readOnlyFields = currentTable?.readOnlyFields || ['ID', 'DateCreated', 'Nid', 'Stamp_create', 'Stamp_mod'];
  
  // Fields that should be completely hidden
  // Combine system hidden fields with configured hidden fields
  const systemHiddenFields = ['@id', '@edit Link', '@context', '_recid'];
  const configuredHiddenFields = currentTable?.hiddenFields || [];
  const hiddenFields = [...systemHiddenFields, ...configuredHiddenFields];
  
  // Convert record data to form fields
  const convertRecordToFormFields = (data: RecordData) => {
    if (!data) {
      setFormFields([]);
      return;
    }
    
    // Use displayShort fields from the current table configuration
    const displayFields = currentTable?.displayShort || [
      'z_recid',
      'Name',
      'StartDate',
      'pl_Status',
      'pl_Cat',
      'Message'
    ];
    
    // Get all keys from the record, excluding OData metadata fields and hidden fields
    const keys = Object.keys(data).filter(key => 
      !key.startsWith('@odata') && 
      !hiddenFields.includes(key) &&
      displayFields.includes(key) // Only include fields in displayShort
    );
    
    // Get field metadata from current table configuration if available
    const fieldConfigs = currentTable?.fields || [];
    
    const fields: FormField[] = [];
    
    // Add fields from record that are in displayShort
    keys.forEach(key => {
      // Find field config if available
      const fieldConfig = fieldConfigs.find(f => f.fieldName === key);
      
      // Determine if this field should be editable
      const isEditable = editableFields.includes(key);
      
      // Special handling for specific fields
      if (key === 'pl_Cat') {
        fields.push({
          key,
          value: data[key] === null ? 'MyStuff' : data[key],
          type: 'select',
          required: false,
          displayName: 'Category',
          options: ['MyStuff', 'Work', 'Personal', 'Health', 'Relax']
        });
      } else if (key === 'Message') {
        fields.push({
          key,
          value: data[key] === null ? '' : data[key],
          type: 'textarea',
          required: true,
          displayName: 'Message',
          multiline: true,
          rows: 4
        });
      } else if (key === 'StartTime') {
        fields.push({
          key,
          value: data[key] === null ? '' : formatTime(data[key]),
          type: 'time',
          inputmode: 'none', // Prevent mobile keyboard
          required: false,
          displayName: 'Start Time'
        });
      } else if (key === 'EndTime') {
        fields.push({
          key,
          value: data[key] === null ? '' : formatTime(data[key]),
          type: 'time',
          required: false,
          displayName: 'End Time'
        });
      } else {
        fields.push({
          key,
          value: data[key] === null ? '' : data[key],
          type: isEditable ? getFieldType(key, data[key], fieldConfig?.type) : 'readonly',
          required: isEditable && (fieldConfig?.required || false),
          displayName: fieldConfig?.displayName || formatFieldName(key)
        });
      }
    });
    
    // Add any configured editable fields from displayShort that weren't in the record
    displayFields.forEach(fieldName => {
      if (editableFields.includes(fieldName) && 
          !hiddenFields.includes(fieldName) && 
          !fields.some(f => f.key === fieldName)) {
        const fieldConfig = fieldConfigs.find(f => f.fieldName === fieldName);
        
        // Special handling for specific fields
        if (fieldName === 'pl_Cat') {
          fields.push({
            key: fieldName,
            value: 'MyStuff', // Default value
            type: 'select',
            required: false,
            displayName: 'Category',
            options: ['MyStuff', 'Work', 'Personal', 'Health', 'Relax']
          });
        } else if (fieldName === 'StartTime') {
          fields.push({
            key: fieldName,
            value: '',
            type: 'time',
            inputmode: 'none', // Prevent mobile keyboard
            required: false,
            displayName: 'Start Time'
          });
        } else if (fieldName === 'EndTime') {
          fields.push({
            key: fieldName,
            value: '',
            type: 'time',
            required: false,
            displayName: 'End Time'
          });
        } else if (fieldName === 'Message') {
          fields.push({
            key: fieldName,
            value: '',
            type: 'textarea',
            required: true,
            displayName: 'Message',
            multiline: true,
            rows: 4
          });
        } else {
          fields.push({
            key: fieldName,
            value: '',
            type: getFieldType(fieldName, '', fieldConfig?.type),
            required: fieldConfig?.required || false,
            displayName: fieldConfig?.displayName || formatFieldName(fieldName)
          });
        }
      }
    });
    
    // Sort fields to match the order in displayShort
    fields.sort((a, b) => {
      const aIndex = displayFields.indexOf(a.key);
      const bIndex = displayFields.indexOf(b.key);
      
      // If both fields are in displayShort, sort by their order there
      if (aIndex >= 0 && bIndex >= 0) {
        return aIndex - bIndex;
      }
      
      // If only one field is in displayShort, prioritize it
      if (aIndex >= 0) return -1;
      if (bIndex >= 0) return 1;
      
      // Otherwise sort alphabetically
      return a.key.localeCompare(b.key);
    });
    
    setFormFields(fields);
  };

  // Determine field type based on key name and value
  const getFieldType = (key: string, value: any, configType?: string): string => {
    // Use configuration type if available
    if (configType) return configType;
    
    // Otherwise infer from key name and value
    const keyLower = key.toLowerCase();
    
    if (keyLower.includes('date')) return 'date';
    if (keyLower.includes('time')) return 'time';
    if (keyLower.includes('email')) return 'email';
    if (keyLower.includes('phone')) return 'tel';
    if (keyLower.includes('url')) return 'url';
    if (keyLower.includes('password')) return 'password';
    
    if (typeof value === 'number') return 'number';
    if (typeof value === 'boolean') return 'checkbox';
    
    if (keyLower === 'status' || keyLower === 'pri' || keyLower === 'priority' || keyLower === 'pl_status' || keyLower === 'pl_cat') return 'select';
    
    // Handle multiline text fields
    if (keyLower === 'message' || keyLower === 'notes' || keyLower === 'description' || keyLower.includes('comment')) {
      return 'textarea';
    }
    
    return 'text';
  };

  // Format field name for display (e.g., 'firstName' -> 'First Name')
  const formatFieldName = (field: string): string => {
    return field
      .replace(/([A-Z])/g, ' $1')
      .replace(/^./, str => str.toUpperCase())
      .trim();
  };

  // Handle form field changes
  const handleFieldChange = (index: number, value: any) => {
    const updatedFields = [...formFields];
    updatedFields[index].value = value;
    setFormFields(updatedFields);
  };

  // Format time to ensure it has AM/PM if it's a valid time
  const formatTime = (timeStr: string) => {
    if (!timeStr) return '';
    
    // If it already has AM/PM, return as is
    if (timeStr.includes('AM') || timeStr.includes('PM')) {
      return timeStr;
    }
    
    // Try to parse and format the time
    const timeMatch = timeStr.match(/^(\d{1,2}):?(\d{2})?/);
    if (timeMatch) {
      let hours = parseInt(timeMatch[1], 10);
      const minutes = timeMatch[2] ? timeMatch[2].padStart(2, '0') : '00';
      const ampm = hours >= 12 ? 'PM' : 'AM';
      hours = hours % 12;
      hours = hours || 12; // Convert 0 to 12 for 12-hour format
      return `${hours}:${minutes} ${ampm}`;
    }
    return timeStr; // Return as is if we can't parse it
  };

  // Handle form submission
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setSaving(true);
    setError('');
    setSaveSuccess(false);
    
    try {
      // Convert form fields to record object - only include editable fields
      const updatedRecord: Record<string, any> = {};
      
      // Get time values for formatting
      const startTimeField = formFields.find(f => f.key === 'StartTime');
      const endTimeField = formFields.find(f => f.key === 'EndTime');
      
      formFields.forEach(field => {
        if (editableFields.includes(field.key)) {
          // Special handling for time fields
          if (field.key === 'StartTime' || field.key === 'EndTime') {
            updatedRecord[field.key] = formatTime(field.value);
          } else {
            updatedRecord[field.key] = field.value;
          }
        }
      });
      
      // Extra safety check - remove any OData metadata fields
      Object.keys(updatedRecord).forEach(key => {
        if (key.startsWith('@') || key === 'context') {
          delete updatedRecord[key];
        }
      });
      
      const encodedDbName = encodeURIComponent(currentDatabaseName);
      const encodedTableName = encodeURIComponent(currentTableName);
      let response;
      let method;
      let url;
      
      if (recordId) {
        // Update existing record
        method = 'PATCH';
        url = `/api/odata/${encodedDbName}/${encodedTableName}/${recordId}`;
      } else {
        // Create new record
        method = 'POST';
        url = `/api/odata/${encodedDbName}/${encodedTableName}`;
      }
      
      const headers: Record<string, string> = {
        'X-Host-Id': currentHostId,
        'Accept': 'application/json',
        'Content-Type': 'application/json',
        'OData-Version': '4.0',
        'OData-MaxVersion': '4.0'
      };
      
      // Add Authorization header if needed (handled by API route)
      
      response = await fetch(url, {
        method,
        headers,
        body: JSON.stringify(updatedRecord),
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || `Error ${response.status}: ${response.statusText}`);
      }
      
      const result = await response.json();
      setSaveSuccess(true);
      
      // Redirect to the task view page after successful save
      setTimeout(() => {
        const newRecordId = recordId || (result && result._recid);
        if (newRecordId) {
          router.push(`/task?tid=${newRecordId}`);
        } else {
          router.push('/tasks-all');
        }
      }, 1500);
     
    } catch (err) {
      console.error('Error saving record:', err);
      setError(err instanceof Error ? err.message : 'An unknown error occurred');
    } finally {
      setSaving(false);
    }
  };

  // Get options for select fields
  const getSelectOptions = (fieldKey: string): string[] => {
    const keyLower = fieldKey.toLowerCase();
    
    if (keyLower === 'status' || keyLower === 'pl_status') {
      return [...statusOptions];
    }
    
    if (keyLower === 'pl_cat') {
      return [...categoryOptions];
    }
    
    if (keyLower === 'notetype_c' || keyLower === 'notetype') {
      return [...noteTypeOptions];
    }
    
    if (keyLower === 'pri' || keyLower === 'priority' || keyLower === 'pl_pri') {
      return [...priorityOptions];
    }
    
    if (keyLower === 'pl_topic' || keyLower === 'topic') {
      return [...topicOptions];
    }
    
    // Check if field has options in table configuration
    const fieldConfig = currentTable?.fields.find(f => f.fieldName === fieldKey);
    if (fieldConfig?.options) {
      return fieldConfig.options;
    }
    
    return [];
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-100">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="bg-white rounded-lg shadow-lg p-6">
            <div className="animate-pulse flex space-x-4">
              <div className="flex-1 space-y-6 py-1">
                <div className="h-4 bg-gray-200 rounded w-3/4"></div>
                <div className="space-y-3">
                  <div className="grid grid-cols-3 gap-4">
                    <div className="h-4 bg-gray-200 rounded col-span-2"></div>
                    <div className="h-4 bg-gray-200 rounded col-span-1"></div>
                  </div>
                  <div className="h-4 bg-gray-200 rounded"></div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-100">
      <NavBar />
      
      {/* Client information display from session storage */}
      {(clientName || storedClientId) && (
        <div className="max-w-7xl mx-auto pt-4 pb-0 sm:px-6 lg:px-8">
          <div className="bg-blue-50 rounded-lg shadow-md p-4 mb-4">
            <div className="text-center">
              <h2 className="text-2xl font-bold text-blue-800">
                Client: {clientName || 'Unknown'}
              </h2>
              <p className="text-lg text-blue-600">
                Client ID: {storedClientId || 'Not specified'}
              </p>
            </div>
          </div>
        </div>
      )}
      
      <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
        <div className="bg-white rounded-lg shadow-lg overflow-hidden">
          <div className="px-3 sm:px-6 py-4 border-b border-gray-200">
            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-3">
              <h1 className="text-xl font-semibold text-gray-900">
                {recordId ? 'Update Record' : 'Create New Record'}
              </h1>
              <div className="flex gap-2">
                <Link
                  href={recordId ? `/task?tid=${recordId}` : '/list'}
                  className="px-3 py-1 bg-gray-100 text-gray-700 rounded hover:bg-gray-200 text-sm"
                >
                  Cancel
                </Link>
              </div>
            </div>
          </div>
          
          <div className="p-1 sm:p-3 md:p-4 database-content">
            {error && (
              <div className="bg-red-50 border-l-4 border-red-400 p-4 my-4">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-red-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.28 7.22a.75.75 0 00-1.06 1.06L8.94 10l-1.72 1.72a.75.75 0 101.06 1.06L10 11.06l1.72 1.72a.75.75 0 101.06-1.06L11.06 10l1.72-1.72a.75.75 0 00-1.06-1.06L10 8.94 8.28 7.22z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-red-700">{error}</p>
                  </div>
                </div>
              </div>
            )}
            
            {saveSuccess && (
              <div className="bg-green-50 border-l-4 border-green-400 p-4 my-4">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-green-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-green-700">Record saved successfully! Redirecting...</p>
                  </div>
                </div>
              </div>
            )}
            
            <form onSubmit={handleSubmit} className="mt-6 space-y-6">
              {/* Editable Fields Section */}
              <div className="mb-8">
                <h2 className="text-lg font-medium text-gray-900 mb-4 pb-2 border-b border-gray-200">Editable Fields</h2>
                <div className="space-y-4">
                  {formFields
                    .filter(field => editableFields.includes(field.key))
                    .map((field) => {
                      const originalIndex = formFields.findIndex(f => f.key === field.key);
                      return (
                        <div key={field.key} className={field.readOnly ? 'opacity-75' : ''}>
                          <label htmlFor={field.key} className="block font-medium text-gray-700">
                            {field.displayName || formatFieldName(field.key)}
                            {field.required && <span className="text-red-500 ml-1">*</span>}
                          </label>
                          <div className="mt-1 mb-4">
                            {field.type === 'select' ? (
                              <select
                                id={field.key}
                                name={field.key}
                                value={field.value || ''}
                                onChange={(e) => handleFieldChange(originalIndex, e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm text-black"
                                disabled={field.readOnly || saving}
                                required={field.required}
                              >
                                <option value="">-- Select --</option>
                                {getSelectOptions(field.key).map(option => (
                                  <option key={option} value={option}>{option}</option>
                                ))}
                              </select>
                            ) : field.type === 'date' ? (
                              <input
                                type="date"
                                id={field.key}
                                name={field.key}
                                value={field.value || ''}
                                onChange={(e) => handleFieldChange(originalIndex, e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm text-black"
                                disabled={field.readOnly || saving}
                                required={field.required}
                              />
                            ) : field.type === 'time' ? (
                              <div className="relative">
                                <input
                                  type="time"
                                  id={field.key}
                                  name={field.key}
                                  value={(() => {
                                    // Convert stored time (e.g., "2:30 PM") back to 24h format for the input
                                    const timeStr = field.value || '';
                                    if (!timeStr) return '';
                                    
                                    const [time, period] = timeStr.split(' ');
                                    if (!time || !period) return '';
                                    
                                    let [hours, minutes] = time.split(':');
                                    if (period === 'PM' && hours !== '12') {
                                      hours = String(parseInt(hours, 10) + 12);
                                    } else if (period === 'AM' && hours === '12') {
                                      hours = '00';
                                    }
                                    return `${hours.padStart(2, '0')}:${minutes || '00'}`;
                                  })()}
                                  onChange={(e) => {
                                    const timeValue = e.target.value;
                                    if (timeValue) {
                                      const [hours, minutes] = timeValue.split(':');
                                      const hoursNum = parseInt(hours, 10);
                                      const ampm = hoursNum >= 12 ? 'PM' : 'AM';
                                      const hours12 = hoursNum % 12 || 12;
                                      const formattedTime = `${hours12}:${minutes || '00'} ${ampm}`;
                                      handleFieldChange(originalIndex, formattedTime);
                                    } else {
                                      handleFieldChange(originalIndex, '');
                                    }
                                  }}
                                  className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm text-black"
                                  disabled={field.readOnly || saving}
                                  required={field.required}
                                />
                                <div 
                                  className="absolute inset-y-0 right-0 flex items-center pr-3 cursor-pointer"
                                  onClick={(e) => {
                                    e.preventDefault();
                                    const input = document.getElementById(field.key) as HTMLInputElement;
                                    if (input) input.showPicker();
                                  }}
                                >
                                  <svg className="h-5 w-5 text-gray-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                    <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clipRule="evenodd" />
                                  </svg>
                                </div>
                              </div>
                            ) : field.multiline ? (
                              <textarea
                                id={field.key}
                                name={field.key}
                                rows={field.rows || 4}
                                value={field.value || ''}
                                onChange={(e) => handleFieldChange(originalIndex, e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm text-black"

                                style={field.key === 'Message' ? { fontSize: '20px', lineHeight: '1.5' } : undefined}


                                disabled={field.readOnly || saving}
                                required={field.required}
                              />
                            ) : field.type === 'checkbox' ? (
                              <div className="flex items-center">
                                <input
                                  id={field.key}
                                  type="checkbox"
                                  checked={field.value}
                                  onChange={(e) => handleFieldChange(originalIndex, e.target.checked)}
                                  className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                  disabled={field.readOnly || saving}
                                />
                                <label htmlFor={field.key} className="ml-2 block text-sm text-gray-900">
                                  {field.value ? 'Yes' : 'No'}
                                </label>
                              </div>
                            ) : (
                              <input
                                type={field.type}
                                id={field.key}
                                name={field.key}
                                value={field.value || ''}
                                onChange={(e) => handleFieldChange(originalIndex, e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm text-black"
                                disabled={field.readOnly || saving}
                                required={field.required}
                              />
                            )}
                          </div>
                        </div>
                      );
                    })
                  }
                </div>
              </div>
              
              {/* Read-Only Fields Section */}
              <div>
                <h2 className="text-lg font-medium text-gray-900 mb-4 pb-2 border-b border-gray-200">Read-Only Fields</h2>
                <div className="space-y-4 sm:space-y-2">
                  {formFields
                    .filter(field => !editableFields.includes(field.key) && !hiddenFields.includes(field.key))
                    .map((field) => (
                      <div key={field.key} className="grid grid-cols-1 gap-2">
                        <label htmlFor={field.key} className="block font-medium text-gray-500 text-sm sm:text-base">
                          {formatFieldName(field.key)}
                        </label>
                        <div className="block w-full rounded-md border border-gray-200 bg-gray-50 py-2 px-3 text-gray-700">
                          {typeof field.value === 'object' ? JSON.stringify(field.value) : 
                           field.value === '' ? '-' : String(field.value)}
                        </div>
                      </div>
                    ))
                  }
                </div>
              </div>
              
              
              <div className="flex justify-end pt-4">
                <button
                  type="submit"
                  className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-opacity-50"
                  disabled={saving}
                >
                  {saving ? 'Saving...' : recordId ? 'Update' : 'Create'}
                </button>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  );
}

// Export the main component wrapped in Suspense
export default function UpdateRecordPage() {
  return (
    <Suspense fallback={
      <div className="min-h-screen bg-gray-50">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="bg-white shadow rounded-lg p-6">
              <h1 className="text-2xl font-bold mb-6">Update Task</h1>
              <p>Loading task form...</p>
            </div>
          </div>
        </div>
      </div>
    }>
      <TaskUpdateContent />
    </Suspense>
  );
}
