'use client';

import { useEffect } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import { useDatabaseContext } from '@/lib/context/DatabaseContext';
import { hosts } from '@/lib/config/databases';

export default function HostSelector() {
  const { currentHostId, setCurrentHostId } = useDatabaseContext();
  const router = useRouter();
  const searchParams = useSearchParams();

  // Handle host selection
  const handleHostChange = (hostId: string) => {
    console.log(`Switching to host: ${hostId}`);
    setCurrentHostId(hostId);
    
    // Update URL to reflect the selected host
    const params = new URLSearchParams(searchParams.toString());
    params.set('host', hostId);
    
    // Navigate to the same page but with updated query params
    router.push(`/?${params.toString()}`);
  };

  // Initialize from URL params if available
  useEffect(() => {
    const hostParam = searchParams.get('host');
    if (hostParam && hostParam !== currentHostId) {
      setCurrentHostId(hostParam);
    }
  }, [searchParams, currentHostId, setCurrentHostId]);

  return (
    <div className="mb-8 p-4 bg-white rounded-lg shadow-md">
      <h2 className="text-xl font-semibold mb-4">Select FileMaker Host</h2>
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        {hosts.map((host) => (
          <button
            key={host.hostId}
            onClick={() => handleHostChange(host.hostId)}
            className={`p-4 rounded-lg transition-colors ${
              currentHostId === host.hostId
                ? 'bg-blue-600 text-white'
                : 'bg-gray-100 hover:bg-gray-200 text-gray-800'
            }`}
          >
            <div className="font-medium">{host.displayName}</div>
            <div className="text-sm mt-1 truncate">
              {currentHostId === host.hostId ? 'Currently selected' : 'Click to select'}
            </div>
          </button>
        ))}
      </div>
    </div>
  );
}
