'use client';

import { useState } from 'react';

interface ScriptDemoProps {
  hostId: string;
  databaseName: string;
  tableName: string;
}

export default function ScriptDemo({ hostId, databaseName, tableName }: ScriptDemoProps) {
  const [scriptName] = useState('odata_test');
  const [scriptParams, setScriptParams] = useState('{\n  "param1": "value1",\n  "param2": "value2"\n}');
  const [isLoading, setIsLoading] = useState(false);
  const [result, setResult] = useState<string | null>(null);
  const [error, setError] = useState<string | null>(null);

  const runScript = async () => {
    if (!scriptName.trim()) {
      setError('Please enter a script name');
      return;
    }

    setIsLoading(true);
    setError(null);
    setResult(null);

    try {
      const params = scriptParams.trim() ? JSON.parse(scriptParams) : {};
      const queryParams = new URLSearchParams();
      
      // Construct the script URL without query parameters
      const scriptUrl = `/api/odata/${databaseName}/Script.${scriptName}`;
      
      // Log the parameters being sent
      console.log('Sending parameters:', params);
      
      console.log('=== Script Execution Request (POST) ===');
      console.log('Script URL:', scriptUrl);
      console.log('Host ID:', hostId);
      console.log('Database Name:', databaseName);
      console.log('Script Name:', scriptName);
      console.log('Parameters:', params);
      console.log('==============================');
      
      // Use POST method and include parameters in the request body
      const response = await fetch(scriptUrl, {
        method: 'POST',
        headers: {
          'X-Host-Id': hostId,
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(params),
      });

      if (!response.ok) {
        throw new Error(`Error: ${response.status} - ${await response.text()}`);
      }

      const data = await response.json();
      setResult(JSON.stringify(data, null, 2));
    } catch (err) {
      setError(err instanceof Error ? err.message : 'An unknown error occurred');
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="bg-white rounded-lg shadow-md p-6 mb-8 border border-gray-200">
      <h2 className="text-xl font-medium text-gray-800 mb-4">Script Demo</h2>
      <p className="text-gray-600 mb-4">
        Execute a FileMaker script via OData. Script names are case-sensitive.
      </p>
      
      <div className="space-y-4">
        <div>
          <label htmlFor="scriptName" className="block text-sm font-medium text-gray-700 mb-1">
            Script Name
          </label>
          <input
            type="text"
            id="scriptName"
            value={scriptName}
            readOnly
            className="w-full px-3 py-2 border border-gray-300 bg-gray-100 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
          />
        </div>

        <div>
          <label htmlFor="scriptParams" className="block text-sm font-medium text-gray-700 mb-1">
            Parameters (JSON)
          </label>
          <textarea
            id="scriptParams"
            value={scriptParams}
            onChange={(e) => setScriptParams(e.target.value)}
            placeholder="Enter parameters as JSON object"
            rows={5}
            className="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 font-mono text-sm"
          />
        </div>

        <div className="flex items-center">
          <button
            onClick={runScript}
            disabled={isLoading}
            className={`px-4 py-2 rounded-md text-white ${isLoading ? 'bg-blue-400' : 'bg-blue-600 hover:bg-blue-700'} focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500`}
          >
            {isLoading ? 'Running...' : 'Run Script'}
          </button>
        </div>

        {error && (
          <div className="p-3 bg-red-50 border-l-4 border-red-500 text-red-700">
            <p className="font-medium">Error</p>
            <p className="text-sm">{error}</p>
          </div>
        )}

        {result && (
          <div>
            <h3 className="text-sm font-medium text-gray-700 mb-1">Result:</h3>
            <pre className="p-3 bg-gray-50 border border-gray-200 rounded-md overflow-auto text-sm">
              {result}
            </pre>
          </div>
        )}
      </div>
    </div>
  );
}
