/**
 * FileMaker OData API Client
 * Handles communication with FileMaker OData endpoints using hardcoded configurations
 */

import { getHost, getTable, HostConfig, TableConfig } from '../config/databases';

interface FetchOptions {
  method?: 'GET' | 'POST' | 'PATCH' | 'DELETE';
  headers?: Record<string, string>;
  body?: any;
  signal?: AbortSignal;
}

export interface QueryOptions {
  filter?: string;
  select?: string[];
  orderby?: string;
  top?: number;
  skip?: number;
  count?: boolean;
  expand?: string[];
  _odataRawParams?: string; // Special property for raw OData parameters
  [key: string]: any; // Allow any other properties
}

interface ODataResponse<T = any> {
  value: T[];
  '@odata.count'?: number;
  '@odata.nextLink'?: string;
}

export class ODataClient {
  private hostConfig: HostConfig | null = null;
  private hostId: string;
  private abortController: AbortController | null = null;

  constructor(hostId: string) {
    if (!hostId) {
      console.error('ODataClient: No hostId provided');
      throw new Error('Host ID is required to create OData client');
    }

    this.hostId = hostId;
    console.log('ODataClient: Initializing for host:', hostId);
    this.abortController = new AbortController();
  }

  private ensureInitialized(): void {
    if (this.hostConfig) return;

    console.log('ODataClient: Loading configuration for host:', this.hostId);

    // Get the host configuration
    const host = getHost(this.hostId);

    if (!host) {
      const error = new Error(`Host configuration not found for ID: ${this.hostId}`);
      console.error('ODataClient: Host configuration error:', error.message);
      console.error('Please check your database configuration for available hosts');
      throw error;
    }

    this.hostConfig = host;
    console.log('ODataClient: Host configuration loaded:', {
      hostId: host.hostId,
      baseUrl: host.baseUrl,
      apiPath: host.apiPath,
      authType: host.authType,
      hasCredentials: !!host.credentials,
      odataUrl: host.odataUrl
    });
  }

  // Clean up any pending requests
  public abortRequests(): void {
    if (this.abortController) {
      console.log('Aborting pending requests');
      this.abortController.abort();
      this.abortController = new AbortController();
    }
  }

  private getAuthHeaders(): Record<string, string> {
    this.ensureInitialized();

    if (!this.hostConfig) {
      throw new Error('Host configuration not initialized');
    }

    const envUsername = process.env.FILEMAKER_USERNAME1;
    const envPassword = process.env.FILEMAKER_PASSWORD1;

    // Use environment variables if available, otherwise fall back to config
    const username = envUsername || this.hostConfig.credentials?.username;
    const password = envPassword || this.hostConfig.credentials?.password;

    if (!username || !password) {
      console.error('Missing credentials for host:', this.hostId);
      throw new Error('Authentication credentials not found');
    }

    // Create Basic Auth token
    const token = Buffer.from(`${username}:${password}`).toString('base64');

    return {
      'Authorization': `Basic ${token}`,
      'Accept': 'application/json',
      'Content-Type': 'application/json',
      'OData-Version': '4.0',
      'OData-MaxVersion': '4.0'
    };
  }

  private buildUrl(databaseName: string, tableName: string, id?: string, queryOptions?: QueryOptions): string {
    this.ensureInitialized();

    if (!this.hostConfig) {
      throw new Error('ODataClient not properly initialized');
    }

    // Use the OData URL from environment variable or construct it from base URL and path
    const odataBaseUrl = process.env.FILEMAKER_ODATA_URL1 || `${this.hostConfig.baseUrl}${this.hostConfig.apiPath}/${databaseName}`;
    const endpoint = tableName;
    let url = `${odataBaseUrl}/${endpoint}`;

    if (id) {
      const formattedId = typeof id === 'string' && !/^\d+$/.test(id) ? `'${id}'` : id;
      url += `(${formattedId})`;
      console.log(`[ODataClient] Building URL with ID: ${id}, formatted as: ${formattedId}`);
      console.log(`[ODataClient] Full URL with ID: ${url}`);
    }

    // Check if we have raw OData parameters to use directly
    if (queryOptions && queryOptions._odataRawParams) {
      // Use the raw OData parameters directly without encoding
      url += '?' + queryOptions._odataRawParams;
      console.log(`[ODataClient] Using raw OData parameters: ${queryOptions._odataRawParams}`);
      return url;
    }
    
    // Otherwise, build the query parameters from the queryOptions object
    if (queryOptions) {
      const params: string[] = [];

      if (queryOptions.filter) {
        // Don't encode the filter parameter - FileMaker OData expects it unencoded
        params.push(`$filter=${queryOptions.filter}`);
      }

      if (queryOptions.select && queryOptions.select.length > 0) {
        params.push(`$select=${queryOptions.select.join(',')}`);
      }

      if (queryOptions.orderby) {
        params.push(`$orderby=${queryOptions.orderby}`);
      }

      if (queryOptions.top !== undefined) {
        params.push(`$top=${queryOptions.top}`);
      }

      if (queryOptions.skip !== undefined) {
        params.push(`$skip=${queryOptions.skip}`);
      }

      if (queryOptions.count) {
        params.push('$count=true');
      }

      if (queryOptions.expand && queryOptions.expand.length > 0) {
        params.push(`$expand=${queryOptions.expand.join(',')}`);
      }

      // Add any other parameters that don't start with $ or _
      Object.entries(queryOptions).forEach(([key, value]) => {
        if (!key.startsWith('$') && !key.startsWith('_') && value !== undefined && 
            key !== 'filter' && key !== 'select' && key !== 'orderby' && 
            key !== 'top' && key !== 'skip' && key !== 'count' && key !== 'expand') {
          params.push(`${key}=${encodeURIComponent(String(value))}`);
        }
      });

      if (params.length > 0) {
        url += '?' + params.join('&');
      }
    }

    return url;
  }

  private async fetchWithAuth<T = any>(url: string, options: FetchOptions = {}): Promise<T> {
    this.ensureInitialized();
    
    if (!this.abortController) {
      this.abortController = new AbortController();
    }

    const headers = {
      ...this.getAuthHeaders(),
      ...(options.headers || {})
    };

    const controller = this.abortController;
    const timeoutId = setTimeout(() => controller.abort(), 30000); // 30 second timeout

    try {
      console.log(`[ODataClient] ${options.method || 'GET'} ${url}`);
      if (options.body) {
        try {
          const body = typeof options.body === 'string' 
            ? JSON.parse(options.body) 
            : options.body;
          console.log('[ODataClient] Request body:', JSON.stringify(body, null, 2));
        } catch (e) {
          console.log('[ODataClient] Request body (raw):', options.body);
        }
      }

      const fetchOptions = {
        ...options,
        headers,
        signal: controller.signal
      };

      const response = await fetch(url, fetchOptions);
      clearTimeout(timeoutId);

      console.log(`[ODataClient] Response status: ${response.status} ${response.statusText}`);

      if (!response.ok) {
        let errorMessage = `HTTP error! status: ${response.status}`;
        let responseBody = '';
        
        try {
          responseBody = await response.text();
          console.log('[ODataClient] Error response body:', responseBody);
          
          if (responseBody) {
            try {
              const errorData = JSON.parse(responseBody);
              errorMessage = errorData.error?.message || 
                           errorData.message || 
                           JSON.stringify(errorData) || 
                           errorMessage;
            } catch (e) {
              errorMessage = `${errorMessage} - ${responseBody}`.trim();
            }
          }
        } catch (e) {
          console.error('[ODataClient] Error reading error response:', e);
          errorMessage = `${errorMessage} - Failed to read response body`;
        }
        
        const error = new Error(errorMessage);
        (error as any).status = response.status;
        (error as any).responseBody = responseBody;
        throw error;
      }

      // For 204 No Content responses, return an empty object
      if (response.status === 204) {
        return {} as T;
      }

      try {
        return await response.json();
      } catch (e) {
        console.error('Error parsing JSON response:', e);
        throw new Error('Failed to parse JSON response');
      }
    } catch (error: unknown) {
      clearTimeout(timeoutId);
      if (error && typeof error === 'object' && 'name' in error) {
        const err = error as { name: string; message?: string; stack?: string };
        if (err.name === 'AbortError') {
          console.error('[ODataClient] Request was aborted (likely due to timeout)');
        }
        console.error('[ODataClient] Fetch error details:', {
          name: err.name,
          message: err.message || 'No error message',
          stack: err.stack
        });
      } else {
        console.error('[ODataClient] Unknown error type:', typeof error, error);
      }
      throw error;
    }
  }

  /**
   * Fetches records from the specified database table
   * @param databaseName Name of the database
   * @param tableName Name of the table
   * @param options Query options (filter, select, orderby, etc.)
   * @returns Promise with the query results and optional count
   */
  async getRecords<T = any>(
    databaseName: string,
    tableName: string,
    options: QueryOptions = {}
  ): Promise<{ data: T[]; count?: number }> {
    console.log(`[ODataClient] Fetching records from ${databaseName}/${tableName}`);
    
    // Log raw OData parameters if present
    if (options._odataRawParams) {
      console.log(`[ODataClient] Using raw OData parameters: ${options._odataRawParams}`);
    } else if (options.filter) {
      console.log(`[ODataClient] Using filter: ${options.filter}`);
    }
    
    const url = this.buildUrl(databaseName, tableName, undefined, options);
    console.log(`[ODataClient] Full URL for getRecords: ${url}`);
    
    const response = await this.fetchWithAuth<ODataResponse<T>>(url);
    
    console.log(`[ODataClient] Received ${response.value?.length || 0} records`);
    if (response['@odata.count'] !== undefined) {
      console.log(`[ODataClient] Total count: ${response['@odata.count']}`);
    }
    
    return {
      data: response.value || [],
      count: response['@odata.count']
    };
  }

  /**
   * Fetches a single record by ID
   * @param databaseName Name of the database
   * @param tableName Name of the table
   * @param id ID of the record to fetch
   * @returns The requested record or null if not found
   */
  async getRecordById<T = any>(
    databaseName: string,
    tableName: string,
    id: string | number
  ): Promise<T | null> {
    try {
      const url = this.buildUrl(databaseName, tableName, String(id));
      const response = await this.fetchWithAuth<T>(url);
      return response || null;
    } catch (error) {
      if ((error as any).status === 404) {
        return null;
      }
      throw error;
    }
  }

  /**
   * Creates a new record
   * @param databaseName Name of the database
   * @param tableName Name of the table
   * @param data The record data to create
   * @returns The created record
   */
  async createRecord<T = any>(
    databaseName: string,
    tableName: string,
    data: Omit<T, 'id'> & { id?: any }
  ): Promise<T> {
    try {
      const url = this.buildUrl(databaseName, tableName);
      const body = typeof data === 'string' ? data : JSON.stringify(data);
      
      console.log(`[ODataClient] Creating record in ${databaseName}/${tableName}`);
      console.log('[ODataClient] Request data:', JSON.stringify(data, null, 2));
      
      const response = await this.fetchWithAuth<T>(url, {
        method: 'POST',
        body,
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        }
      });
      
      return response;
    } catch (error: unknown) {
      console.error('[ODataClient] Error in createRecord:', {
        error: error instanceof Error ? error.message : String(error),
        databaseName,
        tableName,
        data: JSON.stringify(data, (key, value) => 
          typeof value === 'bigint' ? value.toString() : value
        )
      });
      throw error;
    }
  }

  /**
   * Updates an existing record
   * @param databaseName Name of the database
   * @param tableName Name of the table
   * @param id ID of the record to update
   * @param data Partial record data to update
   * @returns Promise that resolves with the updated record or void
   */
  async updateRecord<T = any>(
    databaseName: string,
    tableName: string,
    id: string | number,
    data: Partial<T>
  ): Promise<T | void> {
    try {
      const url = this.buildUrl(databaseName, tableName, String(id));
      const body = typeof data === 'string' ? data : JSON.stringify(data);
      
      console.log(`[ODataClient] Updating record ${id} in ${databaseName}/${tableName}`);
      console.log('[ODataClient] Update data:', JSON.stringify(data, null, 2));
      
      const response = await this.fetchWithAuth<T>(url, {
        method: 'PATCH',
        body,
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        }
      });
      
      return response;
    } catch (error: unknown) {
      console.error('[ODataClient] Error in updateRecord:', {
        error: error instanceof Error ? error.message : String(error),
        databaseName,
        tableName,
        id,
        data: JSON.stringify(data, null, 2)
      });
      throw error;
    }
  }

  /**
   * Deletes a record by ID
   * @param databaseName Name of the database
   * @param tableName Name of the table
   * @param id ID of the record to delete
   * @returns Promise that resolves when the record is deleted
   */
  async deleteRecord(
    databaseName: string,
    tableName: string,
    id: string | number
  ): Promise<void> {
    try {
      const url = this.buildUrl(databaseName, tableName, String(id));
      
      console.log(`[ODataClient] Deleting record ${id} from ${databaseName}/${tableName}`);
      
      await this.fetchWithAuth(url, {
        method: 'DELETE',
        headers: {
          'Accept': 'application/json'
        }
      });
    } catch (error: unknown) {
      console.error('[ODataClient] Error in deleteRecord:', {
        error: error instanceof Error ? error.message : String(error),
        databaseName,
        tableName,
        id
      });
      throw error;
    }
  }
}

// Factory function to create OData clients
export function createODataClient(hostId: string): ODataClient {
  return new ODataClient(hostId);
}

// Default client instance
export const defaultClient = createODataClient('host1');
