'use client';

import React, { createContext, useContext, useState, useEffect, ReactNode } from 'react';
import { 
  hosts, 
  defaultHostId, 
  defaultDatabaseName, 
  defaultTableName,
  HostConfig, 
  DatabaseConfig, 
  TableConfig, 
  getHost, 
  getDatabase, 
  getTable 
} from '../config/databases';
import { ODataClient, createODataClient } from '../api/odataClient';

interface DatabaseContextType {
  // Current selections
  currentHostId: string;
  currentDatabaseName: string;
  currentTableName: string;
  
  // Configuration objects
  currentHost: HostConfig | undefined;
  currentDatabase: DatabaseConfig | undefined;
  currentTable: TableConfig | undefined;
  
  // OData client
  odataClient: ODataClient | undefined;
  
  // Methods to change selections
  setCurrentHostId: (hostId: string) => void;
  setCurrentDatabaseName: (databaseName: string) => void;
  setCurrentTableName: (tableName: string) => void;
  
  // Available options
  availableHosts: HostConfig[];
  availableDatabases: DatabaseConfig[];
  availableTables: TableConfig[];
  
  // Loading state
  isLoading: boolean;
}

const DatabaseContext = createContext<DatabaseContextType | undefined>(undefined);

// Debug function to check environment variables on the client side
function debugEnvironmentVars() {
  console.log('Client-side environment variables:');
  console.log('NEXT_PUBLIC_API_URL:', process.env.NEXT_PUBLIC_API_URL || 'NOT SET');
}

// Re-export default values from the databases config
export { defaultHostId, defaultDatabaseName, defaultTableName };

export function DatabaseProvider({ children }: { children: ReactNode }) {
  // Call debug function
  debugEnvironmentVars();
  
  // State for the current selections
  // Always use host1 since we're using hardcoded configuration
  const [currentHostId, setCurrentHostId] = useState<string>(defaultHostId);
  const [currentDatabaseName, setCurrentDatabaseName] = useState<string>(defaultDatabaseName);
  const [currentTableName, setCurrentTableName] = useState<string>(defaultTableName);
  
  // Process URL parameters on client-side only
  useEffect(() => {
    if (typeof window === 'undefined') return;
    
    const params = new URLSearchParams(window.location.search);
    const tableParam = params.get('table');
    
    console.log('[DatabaseContext] Processing URL parameters:', { tableParam });
    
    // Only set table from URL if we have a valid parameter
    if (tableParam) {
      console.log('[DatabaseContext] Setting table from URL:', tableParam);
      setCurrentTableName(tableParam);
    }
  }, []);
  
  // URL parameter handling has been removed as it's no longer needed
  
  // Derived configuration objects
  const [currentHost, setCurrentHost] = useState<HostConfig | undefined>(undefined);
  const [currentDatabase, setCurrentDatabase] = useState<DatabaseConfig | undefined>(undefined);
  const [currentTable, setCurrentTable] = useState<TableConfig | undefined>(undefined);
  
  // Available options - using hardcoded values
  const [availableHosts] = useState<HostConfig[]>(hosts);
  const [availableDatabases, setAvailableDatabases] = useState<DatabaseConfig[]>([]);
  const [availableTables, setAvailableTables] = useState<TableConfig[]>([]);
  
  // OData client
  const [odataClient, setOdataClient] = useState<ODataClient | undefined>(undefined);
  
  // Loading state
  const [isLoading, setIsLoading] = useState<boolean>(false);
  
  // Initialize OData client and host configuration
  useEffect(() => {
    console.log('[DatabaseContext] Initializing with host ID:', currentHostId);
    const host = getHost(currentHostId);
    console.log('[DatabaseContext] Found host config:', host ? host.hostId : 'Not found');
    
    setCurrentHost(host);
    
    if (!host) {
      console.log('[DatabaseContext] No host found, clearing databases');
      setAvailableDatabases([]);
      setOdataClient(undefined);
      return;
    }
    
    // Update available databases
    console.log('[DatabaseContext] Updating available databases for host:', host.databases.map(d => d.databaseName));
    setAvailableDatabases(host.databases);
    
    // Create OData client
    console.log('[DatabaseContext] Creating OData client for host:', currentHostId);
    const newODataClient = createODataClient(currentHostId);
    setOdataClient(newODataClient);
  }, [currentHostId]);
  
  // Update current database when databaseName changes
  useEffect(() => {
    const database = getDatabase(currentHostId, currentDatabaseName);
    setCurrentDatabase(database);
    
    if (database) {
      setAvailableTables(database.tables);
      const tableExists = database.tables.some(table => table.tableName === currentTableName);
      if (!tableExists && database.tables.length > 0) {
        setCurrentTableName(database.tables[0].tableName);
      }
    } else {
      setAvailableTables([]);
    }
  }, [currentHostId, currentDatabaseName, currentTableName]);
  
  // Update current table when tableName changes
  useEffect(() => {
    const table = getTable(currentHostId, currentDatabaseName, currentTableName);
    setCurrentTable(table);
  }, [currentHostId, currentDatabaseName, currentTableName]);
  
  const value = {
    currentHostId,
    currentDatabaseName,
    currentTableName,
    currentHost,
    currentDatabase,
    currentTable,
    odataClient,
    setCurrentHostId,
    setCurrentDatabaseName,
    setCurrentTableName,
    availableHosts,
    availableDatabases,
    availableTables,
    isLoading,
  };
  
  return (
    <DatabaseContext.Provider value={value}>
      {children}
    </DatabaseContext.Provider>
  );
}

export function useDatabaseContext() {
  const context = useContext(DatabaseContext);
  if (context === undefined) {
    throw new Error('useDatabaseContext must be used within a DatabaseProvider');
  }
  return context;
}
