'use client';

import { useState, useEffect } from 'react';
import { useDatabaseContext } from '../context/DatabaseContext';
import { QueryOptions } from '../api/odataClient';

interface UseODataOptions {
  autoFetch?: boolean;
  queryOptions?: QueryOptions;
}

interface UseODataResult<T> {
  data: T[];
  loading: boolean;
  error: string | null;
  count?: number;
  fetchData: (options?: QueryOptions) => Promise<void>;
  fetchById: (id: string | number) => Promise<T | null>;
  createRecord: (data: Partial<T>) => Promise<T>;
  updateRecord: (id: string | number, data: Partial<T>) => Promise<T | void>;
  deleteRecord: (id: string | number) => Promise<void>;
  refresh: () => Promise<void>;
}

/**
 * Custom hook for interacting with OData endpoints
 * @param options Configuration options
 * @returns Object with data, loading state, error state, and CRUD methods
 */
export function useOData<T = any>(options: UseODataOptions = {}): UseODataResult<T> {
  const { 
    currentHostId, 
    currentDatabaseName, 
    currentTableName,
    odataClient 
  } = useDatabaseContext();
  
  const [data, setData] = useState<T[]>([]);
  const [loading, setLoading] = useState<boolean>(false);
  const [error, setError] = useState<string | null>(null);
  const [count, setCount] = useState<number | undefined>(undefined);
  const [queryOptions, setQueryOptions] = useState<QueryOptions | undefined>(options.queryOptions);
  
  // Function to fetch data
  const fetchData = async (newOptions?: QueryOptions) => {
    if (!odataClient || !currentDatabaseName || !currentTableName) {
      console.error('OData client or database/table not initialized');
      setError('OData client or database/table not initialized');
      return;
    }
    
    try {
      setLoading(true);
      setError(null);
      
      // Use new options if provided, otherwise use the stored options
      const effectiveOptions = newOptions || queryOptions || {};
      
      // Store the new options if provided
      if (newOptions) {
        setQueryOptions(newOptions);
      }
      
      const result = await odataClient.getRecords<T>(
        currentDatabaseName,
        currentTableName,
        effectiveOptions
      );
      
      setData(result.data);
      setCount(result.count);
    } catch (err: any) {
      console.error('Error fetching data:', err);
      setError(err.message || 'Failed to fetch data');
    } finally {
      setLoading(false);
    }
  };
  
  // Function to fetch a record by ID
  const fetchById = async (id: string | number) => {
    if (!odataClient || !currentDatabaseName || !currentTableName) {
      console.error('OData client or database/table not initialized');
      setError('OData client or database/table not initialized');
      return null;
    }
    
    try {
      setLoading(true);
      setError(null);
      
      const result = await odataClient.getRecordById<T>(
        currentDatabaseName,
        currentTableName,
        id
      );
      
      return result;
    } catch (err: any) {
      console.error('Error fetching record by ID:', err);
      setError(err.message || 'Failed to fetch record');
      return null;
    } finally {
      setLoading(false);
    }
  };
  
  // Function to create a record
  const createRecord = async (data: Partial<T>) => {
    if (!odataClient || !currentDatabaseName || !currentTableName) {
      console.error('OData client or database/table not initialized');
      throw new Error('OData client or database/table not initialized');
    }
    
    try {
      setLoading(true);
      setError(null);
      
      const result = await odataClient.createRecord<T>(
        currentDatabaseName,
        currentTableName,
        data as Omit<T, 'id'> & { id?: any }
      );
      
      // Refresh data after creating a record
      await fetchData();
      
      return result;
    } catch (err: any) {
      console.error('Error creating record:', err);
      setError(err.message || 'Failed to create record');
      throw err;
    } finally {
      setLoading(false);
    }
  };
  
  // Function to update a record
  const updateRecord = async (id: string | number, data: Partial<T>) => {
    if (!odataClient || !currentDatabaseName || !currentTableName) {
      console.error('OData client or database/table not initialized');
      throw new Error('OData client or database/table not initialized');
    }
    
    try {
      setLoading(true);
      setError(null);
      
      const result = await odataClient.updateRecord<T>(
        currentDatabaseName,
        currentTableName,
        id,
        data
      );
      
      // Refresh data after updating a record
      await fetchData();
      
      return result;
    } catch (err: any) {
      console.error('Error updating record:', err);
      setError(err.message || 'Failed to update record');
      throw err;
    } finally {
      setLoading(false);
    }
  };
  
  // Function to delete a record
  const deleteRecord = async (id: string | number) => {
    if (!odataClient || !currentDatabaseName || !currentTableName) {
      console.error('OData client or database/table not initialized');
      throw new Error('OData client or database/table not initialized');
    }
    
    try {
      setLoading(true);
      setError(null);
      
      await odataClient.deleteRecord(
        currentDatabaseName,
        currentTableName,
        id
      );
      
      // Refresh data after deleting a record
      await fetchData();
    } catch (err: any) {
      console.error('Error deleting record:', err);
      setError(err.message || 'Failed to delete record');
      throw err;
    } finally {
      setLoading(false);
    }
  };
  
  // Function to refresh data
  const refresh = () => fetchData();
  
  // Auto-fetch data when component mounts or dependencies change
  useEffect(() => {
    if (options.autoFetch !== false && odataClient && currentDatabaseName && currentTableName) {
      fetchData();
    }
  }, [currentHostId, currentDatabaseName, currentTableName]);
  
  return {
    data,
    loading,
    error,
    count,
    fetchData,
    fetchById,
    createRecord,
    updateRecord,
    deleteRecord,
    refresh
  };
}
